﻿<#
.SYNOPSIS
	This script is a template that allows you to extend the toolkit with your own custom functions.
    # LICENSE #
    PowerShell App Deployment Toolkit - Provides a set of functions to perform common application deployment tasks on Windows.
    Copyright (C) 2017 - Sean Lillis, Dan Cunningham, Muhammad Mashwani, Aman Motazedian.
    This program is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free Software Foundation, either version 3 of the License, or any later version. This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
    You should have received a copy of the GNU Lesser General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
.DESCRIPTION
	The script is automatically dot-sourced by the AppDeployToolkitMain.ps1 script.
.NOTES
    Toolkit Exit Code Ranges:
    60000 - 68999: Reserved for built-in exit codes in Deploy-Application.ps1, Deploy-Application.exe, and AppDeployToolkitMain.ps1
    69000 - 69999: Recommended for user customized exit codes in Deploy-Application.ps1
    70000 - 79999: Recommended for user customized exit codes in AppDeployToolkitExtensions.ps1
.LINK
	http://psappdeploytoolkit.com
#>
[CmdletBinding()]
Param (
)

##*===============================================
##* VARIABLE DECLARATION
##*===============================================

# Variables: Script
[string]$appDeployToolkitExtName = 'PSAppDeployToolkitExt'
[string]$appDeployExtScriptFriendlyName = 'App Deploy Toolkit Extensions'
[version]$appDeployExtScriptVersion = [version]'3.8.2'
[string]$appDeployExtScriptDate = '08/05/2020'
[hashtable]$appDeployExtScriptParameters = $PSBoundParameters

##*===============================================
##* FUNCTION LISTINGS
##*===============================================

# <Your custom functions go here>
#region Function Get-Shortcut
function Get-Shortcut {
	<#
	.SYNOPSIS
		This function searches for files matching a LNK and URL extension.
	.DESCRIPTION
		This function, by default, recursively searches for files matching a LNK and URL extensions containing
		a specific string inside the target path, name or both. If no folder path specified, it will 
		recursively search all user profiles and the all users profile.
	.NOTES
		Created on: 	6/23/2014
		Created by: 	Adam Bertram
	.EXAMPLE
		Get-Shortcut -TargetPath 'http:\\servername\local'
		This example would find all shortcuts (URL and LNK) in all user profiles that have a 
		target path that match 'http:\\servername\local'
	.EXAMPLE
		Get-Shortcut -TargetPath 'http:\\servername\local' -Name 'name'
		This example would find all shortcuts (URL and LNK) in all user profiles that have a 
		target path that match 'http:\\servername\local' and have a name containing the string "name"
	.EXAMPLE
		Get-Shortcut -TargetPath 'http:\\servername\local' -FilePath 'C:\Users\abertram\Desktop'
		This example would find all shortcuts (URL and LNK) in the 'C:\Users\abertram\Desktop file path 
		that have a target path that match 'http:\\servername\local' and have a name containing the 
		string "name"
	.PARAMETER TargetPath
		The string you'd like to search for inside the shortcut's target path
	.PARAMETER Name
		A string you'd like to search for inside of the shortcut's name
	.PARAMETER FilePath
		A string you'd like to search for inside of the shortcut's file path
	.PARAMETER FolderPath
		The folder path to search for shortcuts in.  You can specify multiple folder paths. This defaults to 
		the user profile root and the all users profile
	.PARAMETER NoRecurse
		This turns off recursion on the folder path specified searching subfolders of the FolderPath
	#>
	[CmdletBinding()]
	param (
		[string]$TargetPath,
		[string]$Name,
		[string]$FilePath,
		[string[]]$FolderPath,
		[switch]$NoRecurse
	)
	begin {
		function Get-RootUserProfileFolderPath {
			<#
			.SYNOPSIS
				Because sometimes the root user profile folder path can be different this function is a placeholder to find
				the root user profile folder path ie. C:\Users or C:\Documents and Settings for any OS.  It queries a registry value
				to find this path.
			#>
			[CmdletBinding()]
			param ()
			process {
				try {
					(Get-ItemProperty -Path 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList' -Name ProfilesDirectory).ProfilesDirectory
				} catch {
					Write-Log -Message "Error: $($_.Exception.Message) - Line Number: $($_.InvocationInfo.ScriptLineNumber)" -LogLevel '3'
					$false
				}
			}
		}
		function Get-AllUsersProfileFolderPath {
			<#
			.SYNOPSIS
				Because sometimes the all users profile folder path can be different this function is a placeholder to find
				the all users profile folder path ie. C:\ProgramData or C:\Users\All Users. It uses an environment variable
				to find this path.
			#>
			[CmdletBinding()]
			param ()
			process {
				try {
					$env:ALLUSERSPROFILE
				} catch {
					Write-Log -Message "Error: $($_.Exception.Message) - Line Number: $($_.InvocationInfo.ScriptLineNumber)" -LogLevel '3'
					$false
				}
			}
		}
	}
	process {
		try {
			if (!$FolderPath) {
				$FolderPath = (Get-RootUserProfileFolderPath), (Get-AllUsersProfileFolderPath)
			}
			
			$Params = @{
				'Include' = @('*.url', '*.lnk');
				'ErrorAction' = 'SilentlyContinue';
				'ErrorVariable' = 'MyError';
				'Force' = $true
			}
			
			if (!$NoRecurse) {
				$Params['Recurse'] = $true
			}
			
			$ShellObject = New-Object -ComObject Wscript.Shell
			[System.Collections.ArrayList]$Shortcuts = @()
			
			foreach ($Path in $FolderPath) {
				try {
					Write-Verbose -Message "Searching for shortcuts in $Path..."
					[System.Collections.ArrayList]$WhereConditions = @()
					$Params['Path'] = $Path
					if ($TargetPath) {
						$WhereConditions.Add('(($ShellObject.CreateShortcut($_.FullName)).TargetPath -like "*$TargetPath*")') | Out-Null
					}
					if ($Name) {
						$WhereConditions.Add('($_.Name -like "*$Name*")') | Out-Null
					}
					if ($FilePath) {
						$WhereConditions.Add('($_.FullName -like "*$FilePath*")') | Out-Null
					}
					if ($WhereConditions.Count -gt 0) {
						$WhereBlock = [scriptblock]::Create($WhereConditions -join ' -and ')
						Get-ChildItem @Params | where $WhereBlock
					} else {
						Get-ChildItem @Params
					}
					if ($NewShortcuts) {
						$Shortcuts.Add($NewShortcuts) | Out-Null
					}
				} catch {
					Write-Error -Message "Error: $($_.Exception.Message) - Line Number: $($_.InvocationInfo.ScriptLineNumber)"
					$false
				}
			}
		} catch {
			Write-Error -Message "Error: $($_.Exception.Message) - Line Number: $($_.InvocationInfo.ScriptLineNumber)"
			$false
		}
	}
}
#endregion

#region Function Set-Shortcut
function Set-Shortcut {
	<#
	.SYNOPSIS
		This function modifies a LNK or URL extension shortcut.
	.EXAMPLE
		PS> Get-Shortcut -TargetPath 'http:\\servername\local' | Set-Shortcut -TargetPath 'http:\\newserver\local'
		
		This example would find all shortcuts (URL and LNK) in all user profiles that have a 
		target path that match 'http:\\servername\local' and change that target path to 
		'http:\\newserver\local'
	.PARAMETER FilePath
		One or more file paths to the shortcut file
	.PARAMETER TargetPath
		The target path you'd like to set in the shortcut
	.PARAMETER Comment
		The description of the shortcut you'd like to change to
	#>
	[CmdletBinding(DefaultParameterSetName = 'Default')]
	param (
		[Parameter(Mandatory=$true,ValueFromPipeline=$true,ValueFromPipelineByPropertyName=$true)]
		[ValidateScript({ Test-Path -Path $_ -PathType Leaf })]
		[Alias('Fullname')]
		[string[]]$FilePath,
		[Parameter(Mandatory=$false,ParameterSetName = 'TargetPath')]
		[string]$TargetPath,
		[Parameter(Mandatory=$false,ParameterSetName = 'Arguements')]
		[string]$Arguements,
		[Parameter(Mandatory=$false,ParameterSetName = 'IconLocation')]
		[string]$IconLocation,
		[Parameter(Mandatory=$false, ParameterSetName = 'Comment')]
		[string]$Comment
	)
	process {
		try {
			$ShellObject = New-Object -ComObject Wscript.Shell
			foreach ($File in $FilePath) {
				try {
					$Shortcut = $ShellObject.CreateShortcut($File)
					if ($TargetPath) {
						$Shortcut.TargetPath = $TargetPath
					}
					if ($Arguements) {
						$Shortcut.Arguments = $Arguements
					}
					if ($IconLocation) {
						$Shortcut.IconLocation = $IconLocation
					}
					if ($Comment) {
						$Shortcut.Description = $Comment
					}
					$Shortcut.Save()
				} catch {
					Write-Error -Message "Error: $($_.Exception.Message) - Line Number: $($_.InvocationInfo.ScriptLineNumber)"
				}
			}
		} catch {
			Write-Error -Message "Error: $($_.Exception.Message) - Line Number: $($_.InvocationInfo.ScriptLineNumber)"
		}
	}
}
#endregion

#region Function Get-ApplicationMinimumVersion

Function Get-ApplicationMinimumVersion {
	<#
	.SYNOPSIS
		This function searches for the installed application/s and verifies the minimum version is installed. Returns Boolean
	.PARAMETER Name
		The Name of the application
	.PARAMETER MinVersion
		The Minimum Version of the application to check for
	#>
	[CmdletBinding()]
	Param (
		[Parameter(Mandatory=$true)]
		[string[]]$Name,
		[Parameter(Mandatory=$true)]
		[string]$MinVersion
	)
	Process {
		$applications = Get-InstalledApplication -Name $Name
		$result = $false
		Foreach ($application in $applications){
		$appversion = $application.DisplayVersion
			If($appversion -ge $MinVersion){
				$result = $true
				Break
			}
		}
		
		If($result){Return $true}Else{Return $false}
	}
}

#endregion

#region Function CustomInputBox
Function CustomInputBox([string] $title, [string] $message, [string] $defaultText) 
 {
  [void] [System.Reflection.Assembly]::LoadWithPartialName("System.Drawing")
  [void] [System.Reflection.Assembly]::LoadWithPartialName("System.Windows.Forms") 
 
 $userForm = New-Object System.Windows.Forms.Form
  $userForm.Text = "$title"
  $userForm.Size = New-Object System.Drawing.Size(290,150)
  $userForm.StartPosition = "CenterScreen"
      $userForm.AutoSize = $False
      $userForm.MinimizeBox = $False
      $userForm.MaximizeBox = $False
      $userForm.SizeGripStyle= "Hide"
      $userForm.WindowState = "Normal"
      $userForm.FormBorderStyle="Fixed3D"
     
  $OKButton = New-Object System.Windows.Forms.Button
  $OKButton.Location = New-Object System.Drawing.Size(115,80)
  $OKButton.Size = New-Object System.Drawing.Size(75,23)
  $OKButton.Text = "OK"
  $OKButton.Add_Click({$value=$objTextBox.Text;$userForm.Close()})
  $userForm.Controls.Add($OKButton)
 
 $CancelButton = New-Object System.Windows.Forms.Button
  $CancelButton.Location = New-Object System.Drawing.Size(195,80)
  $CancelButton.Size = New-Object System.Drawing.Size(75,23)
  $CancelButton.Text = "Cancel"
  $CancelButton.Add_Click({$userForm.Close()})
  $userForm.Controls.Add($CancelButton)
 
 $userLabel = New-Object System.Windows.Forms.Label
  $userLabel.Location = New-Object System.Drawing.Size(10,20)
  $userLabel.Size = New-Object System.Drawing.Size(280,20)
  $userLabel.Text = "$message"
  $userForm.Controls.Add($userLabel) 
 
 $objTextBox = New-Object System.Windows.Forms.TextBox
  $objTextBox.Location = New-Object System.Drawing.Size(10,40)
  $objTextBox.Size = New-Object System.Drawing.Size(260,20)
  $objTextBox.Text="$defaultText"
  $userForm.Controls.Add($objTextBox) 
 
 $userForm.Topmost = $True
  $userForm.Opacity = 1
      $userForm.ShowIcon = $False
 
 $userForm.Add_Shown({$userForm.Activate()})
  [void] $userForm.ShowDialog()
 
 return $objTextBox.Text 
 }
#endregion

#region Function Get-ComputerOU
Function Get-ComputerOU {

Process {
	$strName = $env:computername
	$strFilter = "(&(objectCategory=Computer)(Name=$strName))"

	$objSearcher = New-Object System.DirectoryServices.DirectorySearcher
	$objSearcher.Filter = $strFilter

	$objPath = $objSearcher.FindOne()
	$objComputer = $objPath.GetDirectoryEntry()
	$computerDN = $objComputer.distinguishedName
	$OU = ($computerDN -replace "^CN=.*?,")
	Return $OU
}
}
#endregion

#region Function Create-HWCustomRegistryFields
Function Create-HWCustomRegistryFields {

		$PackageKeyName = 'SWD-' + $installName + '-' + $deploymentType
				
		If($Is64Bit){$PackageKeyNamePath = "HKLM:\SOFTWARE\WOW6432Node\Honeywell\Custom Fields\$PackageKeyName"}
		Else{$PackageKeyNamePath = "HKLM:\SOFTWARE\Honeywell\Custom Fields\$PackageKeyName"}
		
		#Write-Host "PackageKeyPath = $PackageKeyNamePath"
		
		If(-not (Test-Path $PackageKeyNamePath)){
			#Write-Host "$PackageKeyNamePath does not exist"
			Set-RegistryKey -Key $PackageKeyNamePath
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'StartTime' -Value "$StartTime"
			$AttemptsValue = 1
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'Attempts' -Value "$AttemptsValue"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'ESD Number' -Value "$ESDNumber"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'OS' -Value "$envOSName $envOSServicePack $envOSArchitecture $envOSVersion"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'Run_By' -Value "$ProcessNTAccount"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'UserName' -Value "$($CurrentLoggedOnUserSession.UserName)"
            Set-RegistryKey -Key $PackageKeyNamePath -Name 'Status' -Value ""
			}
		Else {
			#Write-Host "$PackageKeyNamePath does exist"
			$PreviousAttempts = (Get-ItemProperty $PackageKeyNamePath).Attempts
			$AttemptsValue = ([INT]$PreviousAttempts + 1)
			#Write-Host "Package $PackageKeyName has been Attempted $PreviousAttempts times"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'StartTime' -Value "$StartTime"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'Attempts' -Value "$AttemptsValue"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'ESD Number' -Value "$ESDNumber"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'OS' -Value "$envOSName $envOSServicePack $envOSArchitecture $envOSVersion"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'Run_By' -Value "$ProcessNTAccount"
			Set-RegistryKey -Key $PackageKeyNamePath -Name 'UserName' -Value "$($CurrentLoggedOnUserSession.UserName)"
            Set-RegistryKey -Key $PackageKeyNamePath -Name 'Status' -Value ""
			}
			
		Return($AttemptsValue)
	}
#endregion

#region Function Remove-HWCustomRegistryFields
Function Remove-HWCustomRegistryFields {
	$PackageKeyName = 'SWD-' + $installName + '-Install'
	
	If($Is64Bit){$PackageKeyNamePath = "HKLM:\SOFTWARE\WOW6432Node\Honeywell\Custom Fields\$PackageKeyName"}
	Else{$PackageKeyNamePath = "HKLM:\SOFTWARE\Honeywell\Custom Fields\$PackageKeyName"}
	
	If(Test-Path $PackageKeyNamePath){Remove-RegistryKey -Key $PackageKeyNamePath }
}
#endregion

#region Function Update-HWCustomRegistryFields
Function Update-HWCustomRegistryFields([string] $Status, [string] $FINALMSG, [Int32]$ExitCode) {
	$PackageKeyName = 'SWD-' + $installName + '-' + $deploymentType
	
	If($Is64Bit){$PackageKeyNamePath = "HKLM:\SOFTWARE\WOW6432Node\Honeywell\Custom Fields\$PackageKeyName"}
	Else{$PackageKeyNamePath = "HKLM:\SOFTWARE\Honeywell\Custom Fields\$PackageKeyName"}
	
	If(Test-Path $PackageKeyNamePath){
    Set-RegistryKey -Key $PackageKeyNamePath -Name 'LastStatus' -Value "$FINALMSG"
    Write-Host "$FINALMSG"
	Set-RegistryKey -Key $PackageKeyNamePath -Name 'ExitCode' -Value "$ExitCode"
    Set-RegistryKey -Key $PackageKeyNamePath -Name 'Status' -Value "$Status"
    Write-Host "$Status"
	}
}
#endregion

#region Function Test-VPNConnection
Function Test-VPNConnection
{
<#
.SYNOPSIS
    Check to see if there is an active VPN connection.
    
.DESCRIPTION
    Check to see if there is an active VPN connection by using the Win32_NetworkAdapter and the
     Win32_NetworkAdapterConfiguration WMI classes.
    
.PARAMETER NotMatchAdapterDescription
    Excludes on the network adapter description field using regex matching. Precedence order: 0.
    Following WAN Miniport adapters are used for Microsoft Remote Access based VPN
     so are not excluded by default: L2TP, SSTP, IKEv2, PPTP
    
.PARAMETER LikeAdapterDescription
    Matches on the network adapter description field using wild card matching. Precedence order: 1.
    
.PARAMETER LikeAdapterDNSDomain
    Matches on the network adapter DNS Domain field using wild card matching. Precedence order: 2.
    
.PARAMETER LikeAdapterDHCPServer
    Matches on the network adapter DHCP Server field using wild card matching. Precedence order: 3.
    
.PARAMETER LikeAdapterDefaultGateway
    Matches on the network adapter Default Gateway field using wild card matching. Precedence order: 4.
    
.PARAMETER DisplayNetworkAdapterTable
    Logs the full list of network adapters and also the filterd list of possible VPN connection
     network adapters.
    
.EXAMPLE
    Test-VPNConnection
    
.NOTES
    $AllNetworkAdapterConfigTable contains all criteria for detecting VPN connections.
    Try to choose criteria that:
      1) Uniquely identifies the network(s) of interest.
      2) Try not to rely on networking data that may change in future. For example, default gateways
         and DNS and DHCP addresses may change over time or there may be too many to match on.
         Try to use wildcard or regular expression matches if there is an available pattern
         to match multiple values on.
.LINK
#>
	[CmdletBinding()]
	Param
	(
		[Parameter(Mandatory = $false)]
		[ValidateNotNullorEmpty()]
		[string[]]$NotMatchAdapterDescription = ('^WAN Miniport \(PPPOE\)', '^WAN Miniport \(IPv6\)', '^WAN Miniport \(Network Monitor\)',
			'^WAN Miniport \(IP\)', '^Microsoft 6to4 Adapter', '^Microsoft Virtual WiFi Miniport Adapter',
			'^Microsoft WiFi Direct Virtual Adapter', '^Microsoft ISATAP Adapter', '^Direct Parallel',
			'^Microsoft Kernel Debug Network Adapter', '^Microsoft Teredo', '^Packet Scheduler Miniport',
			'^VMware Virtual', '^vmxnet', 'VirtualBox', '^Bluetooth Device', '^RAS Async Adapter', 'USB'),
		[Parameter(Mandatory = $false)]
		[ValidateNotNullorEmpty()]
		[string[]]$LikeAdapterDescription = ('*vpn*', '*juniper*', '*check point*', '*cisco anyconnect*'),
		[Parameter(Mandatory = $false)]
		[ValidateNotNullorEmpty()]
		[string[]]$LikeAdapterDNSDomain,
		[Parameter(Mandatory = $false)]
		[ValidateNotNullorEmpty()]
		[string[]]$LikeAdapterDHCPServer,
		[Parameter(Mandatory = $false)]
		[ValidateNotNullorEmpty()]
		[string[]]$LikeAdapterDefaultGateway,
		[Parameter(Mandatory = $false)]
		[switch]$DisplayNetworkAdapterTable = $false
	)
	
	Begin
	{
		[scriptblock]$AdapterDescriptionFilter = {
			[CmdletBinding()]
			Param
			(
				[Parameter(Mandatory = $true, Position = 0, ValueFromPipeline = $true, ValueFromPipelineByPropertyName = $true)]
				$sbInputObject,
				[Parameter(Mandatory = $true, Position = 1)]
				[string[]]$sbNotMatchAdapterDescription
			)
			
			$SendToPipeline = $true
			ForEach ($sbNotMatchDesc in $sbNotMatchAdapterDescription)
			{
				If ($sbInputObject.Description -imatch $sbNotMatchDesc)
				{
					$SendToPipeline = $false
					Break
				}
			}
			
			If ($SendToPipeline)
			{
				Write-Output $sbInputObject
			}
		}
	}
	Process
	{
		Try
		{
			[psobject[]]$AllNetworkAdapter = Get-WmiObject Win32_NetworkAdapter -ErrorAction 'Stop' |
			Select-Object -Property DeviceID, PNPDeviceID, Manufacturer
			
			[psobject[]]$AllNetworkAdapterConfigTemp = Get-WmiObject Win32_NetworkAdapterConfiguration -ErrorAction 'Stop' |
			Select-Object -Property @{ L = 'DeviceID'; E = { $_.Index } }, DNSDomain, DefaultIPGateway, DHCPServer, IPEnabled, PhysicalAdapter, Manufacturer, Description
			
			ForEach ($AdapterConfig in $AllNetworkAdapterConfigTemp)
			{
				ForEach ($Adapter in $AllNetworkAdapter)
				{
					If ($AdapterConfig.DeviceID -eq $Adapter.DeviceID)
					{
						## Note: We create our own custom PhysicalAdapter property b/c the one in the
						##       Win32_NetworkAdapter class is not accurate.
						$AdapterConfig.PhysicalAdapter = [boolean]($Adapter.PNPDeviceID -imatch '^PCI\\')
						$AdapterConfig.Manufacturer = $Adapter.Manufacturer
						[psobject[]]$AllNetworkAdapterConfig += $AdapterConfig
					}
				}
			}
			
			## This table contains the major markers that might help user create the criteria for detecting VPN connections.
			[string]$AllNetworkAdapterConfigTable = $AllNetworkAdapterConfig |
			Format-Table DNSDomain, DefaultIPGateway, DHCPServer, IPEnabled, PhysicalAdapter, Manufacturer, Description -AutoSize -Wrap | Out-String
			
			## Sanitize list of Network Adapters by removing:
			##  a) physical adapters
			##  b) adapters which we know are not VPN connections
			[psobject[]]$NetworkAdapterConfig = $AllNetworkAdapterConfig |
			Where-Object   { -not ($_.PhysicalAdapter) } |
			ForEach-Object {
				&$AdapterDescriptionFilter -sbInputObject $_ -sbNotMatchAdapterDescription $NotMatchAdapterDescription
			}
			[string]$NetworkAdapterConfigTable = $NetworkAdapterConfig |
			Format-Table DNSDomain, DefaultIPGateway, DHCPServer, IPEnabled, PhysicalAdapter, Manufacturer, Description -AutoSize -Wrap | Out-String
			
			## Sanitize list of Network Adapters by removing:
			##  a) adapters which are not connected (IP Enabled)
			$NetworkAdapterConfig = $NetworkAdapterConfig | Where-Object { $_.IpEnabled }
			[string]$IpEnabledNetworkAdapterConfigTable = $NetworkAdapterConfig |
			Format-Table DNSDomain, DefaultIPGateway, DHCPServer, IPEnabled, PhysicalAdapter, Manufacturer, Description -AutoSize -Wrap | Out-String
			
			## Discover VPN Network Adapter by using multiple search criteria.
			## Search stops at the first match using below precedence order.
			[string]$VPNMatchUsing = ''
			
			#  Precedence Order 1: Detect VPN connection based on key words in network adapter description field.
			If ($LikeAdapterDescription)
			{
				ForEach ($LikeDescription in $LikeAdapterDescription)
				{
					If ([boolean]($NetworkAdapterConfig | Where-Object { ($_ | Select-Object -ExpandProperty Description) -ilike $LikeDescription }))
					{
						$VPNMatchUsing = 'VPN Network Adapter matched on search criteria in parameter [-LikeAdapterDescription]'
						Return $true
					}
				}
			}
			
			#  Precedence Order 2: Detect VPN based on DNS domain (e.g.: contoso.com).
			If ($LikeAdapterDNSDomain)
			{
				ForEach ($LikeDNSDomain in $LikeAdapterDNSDomain)
				{
					If ([boolean]($NetworkAdapterConfig | Where-Object { ($_ | Select-Object -ExpandProperty DNSDomain) -ilike $LikeDNSDomain }))
					{
						$VPNMatchUsing = 'VPN Network Adapter matched on search criteria in parameter [-LikeAdapterDNSDomain]'
						Return $true
					}
				}
			}
			
			#  Precedence Order 3: Detect VPN connection based on the DHCP Server of the network adapter
			If ($LikeAdapterDHCPServer)
			{
				ForEach ($LikeDHCPServer in $LikeAdapterDHCPServer)
				{
					If ([boolean]($NetworkAdapterConfig | Where-Object { ($_ | Select-Object -ExpandProperty DHCPServer) -ilike $LikeDHCPServer }))
					{
						$VPNMatchUsing = 'VPN Network Adapter matched on search criteria in parameter [-LikeAdapterDHCPServer]'
						Return $true
					}
				}
			}
			
			#  Precedence Order 4: Detect VPN connection based on the default gateway for the network adapter.
			If ($LikeAdapterDefaultGateway)
			{
				ForEach ($LikeDefaultGateway in $LikeAdapterDefaultGateway)
				{
					If ([boolean]($NetworkAdapterConfig | Where-Object { ($_ | Select-Object -ExpandProperty DefaultIPGateway) -ilike $LikeDefaultGateway }))
					{
						$VPNMatchUsing = 'VPN Network Adapter matched on search criteria in parameter [-LikeAdapterDefaultGateway]'
						Return $true
					}
				}
			}
			Return $false
		}
		Catch
		{
			Return $false
		}
	}
	End
	{
		## Display Network Adapter Tables
		If ($DisplayNetworkAdapterTable)
		{
			Write-Host "All network adapters: `n$AllNetworkAdapterConfigTable" -ForegroundColor 'Magenta'
			Write-Host "Filtered to possible VPN network adapters: `n$NetworkAdapterConfigTable" -ForegroundColor 'Yellow'
			Write-Host "Filtered to possible VPN network adapters (IP Enabled): `n$IpEnabledNetworkAdapterConfigTable" -ForegroundColor 'Cyan'
			If (-not ([string]::IsNullOrEmpty($VPNMatchUsing)))
			{
				Write-Host "$VPNMatchUsing" -ForegroundColor 'White'
			}
		}
	}
}
#endregion

#region Function Search-RegistryUninstallKey
function Search-RegistryUninstallKey {
<#
.SYNOPSIS
	Enumerate Windows Uninstall Registry Keys.
.DESCRIPTION
	Allows You to search Uninstall Registy Keys to see if particular software is installed
.PARAMETER SearchCriteria
	Options: GUID, Displayname, DisplayVersion, Publisher
.PARAMETER SearcFor
	What to search for used in conjuction with SearchCriteria, GUID, DisplayName, DisplayVersion, or Publisher
.EXAMPLE
	Search-RegistryUninstallKey -SearchCriteria 'DisplayName' -SearchFor 'Microsoft Office'
	Search-RegistryUninstallKey -SearchCriteria 'Publisher' -SearchFor 'Microsoft'
	Search-RegistryUninstallKey | Out-GridView
#>
[CmdletBinding()]
param($SearchFor,[ValidateSet('GUID','DisplayName','DisplayVersion','Publisher')] [String]$SearchCriteria)
If($((gwmi win32_operatingsystem | select osarchitecture).osarchitecture) -eq "64-bit"){[Bool]$Is64Bit = $true}Else{[Bool]$Is64Bit = $false}
$results = @()
$keys = Get-ChildItem HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall | 
    foreach {
        $obj = New-Object psobject
        Add-Member -InputObject $obj -MemberType NoteProperty -Name GUID -Value $_.pschildname
		Add-Member -InputObject $obj -MemberType NoteProperty -Name Publisher -Value $_.GetValue("Publisher")
        Add-Member -InputObject $obj -MemberType NoteProperty -Name DisplayName -Value $_.GetValue("DisplayName")
        Add-Member -InputObject $obj -MemberType NoteProperty -Name DisplayVersion -Value $_.GetValue("DisplayVersion")
		Add-Member -InputObject $obj -MemberType NoteProperty -Name UninstallString -Value $_.GetValue("UninstallString")
        if ($Wow6432Node)
        {Add-Member -InputObject $obj -MemberType NoteProperty -Name Wow6432Node? -Value "No"}
        $results += $obj
        }
 
if ($Is64Bit) {
$keys = Get-ChildItem HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall | 
    foreach {
        $obj = New-Object psobject
        Add-Member -InputObject $obj -MemberType NoteProperty -Name GUID -Value $_.pschildname
        Add-Member -InputObject $obj -MemberType NoteProperty -Name Publisher -Value $_.GetValue("Publisher")
        Add-Member -InputObject $obj -MemberType NoteProperty -Name DisplayName -Value $_.GetValue("DisplayName")
        Add-Member -InputObject $obj -MemberType NoteProperty -Name DisplayVersion -Value $_.GetValue("DisplayVersion")
		Add-Member -InputObject $obj -MemberType NoteProperty -Name UninstallString -Value $_.GetValue("UninstallString")
        Add-Member -InputObject $obj -MemberType NoteProperty -Name Wow6432Node? -Value "Yes"
        $results += $obj
        }
    }
	
Switch ($SearchCriteria)
	{
		"GUID" {$results | sort DisplayName | where {$_.GUID -match $SearchFor}}
		"DisplayName" {$results | sort DisplayName | where {$_.DisplayName -match $SearchFor}}
		"DisplayVersion" {$results | sort DisplayName | where {$_.DisplayVersion -match $SearchFor}}
		"Publisher" {$results | sort DisplayName | where {$_.Publisher -match $SearchFor}}
		default {$results | sort DisplayName | where {$_.DisplayName -match $SearchFor}}
	}
}
#endregion

#region Function Get-OfficeVersion
Function Get-OfficeVersion
{
<#
.Synopsis
Gets the Office Version installed on the computer
.DESCRIPTION
This function will query the local or a remote computer and return the information about Office Products installed on the computer
.NOTES   
Name: Get-OfficeVersion
Version: 1.0.5
DateCreated: 2015-07-01
DateUpdated: 2016-07-20
.LINK
https://github.com/OfficeDev/Office-IT-Pro-Deployment-Scripts
.PARAMETER ComputerName
The computer or list of computers from which to query 
.PARAMETER ShowAllInstalledProducts
Will expand the output to include all installed Office products
.EXAMPLE
Get-OfficeVersion
Description:
Will return the locally installed Office product
.EXAMPLE
Get-OfficeVersion -ComputerName client01,client02
Description:
Will return the installed Office product on the remote computers
.EXAMPLE
Get-OfficeVersion | select *
Description:
Will return the locally installed Office product with all of the available properties
#>
	[CmdletBinding(SupportsShouldProcess = $true)]
	param (
		[Parameter(ValueFromPipeline = $true, ValueFromPipelineByPropertyName = $true, Position = 0)]
		[string[]]$ComputerName = $env:COMPUTERNAME,
		[switch]$ShowAllInstalledProducts,
		[System.Management.Automation.PSCredential]$Credentials
	)
	
	begin
	{
		$HKLM = [UInt32] "0x80000002"
		$HKCR = [UInt32] "0x80000000"
		
		$excelKeyPath = "Excel\DefaultIcon"
		$wordKeyPath = "Word\DefaultIcon"
		
		$installKeys = 'SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall',
		'SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall'
		
		$officeKeys = 'SOFTWARE\Microsoft\Office',
		'SOFTWARE\Wow6432Node\Microsoft\Office'
		
		$defaultDisplaySet = 'DisplayName', 'Version', 'ComputerName'
		
		$defaultDisplayPropertySet = New-Object System.Management.Automation.PSPropertySet('DefaultDisplayPropertySet', [string[]]$defaultDisplaySet)
		$PSStandardMembers = [System.Management.Automation.PSMemberInfo[]]@($defaultDisplayPropertySet)
	}
	
	process
	{
		
		$results = new-object PSObject[] 0;
		
		foreach ($computer in $ComputerName)
		{
			if ($Credentials)
			{
				$os = Get-WMIObject win32_operatingsystem -computername $computer -Credential $Credentials
			}
			else
			{
				$os = Get-WMIObject win32_operatingsystem -computername $computer
			}
			
			$osArchitecture = $os.OSArchitecture
			
			if ($Credentials)
			{
				$regProv = Get-Wmiobject -list "StdRegProv" -namespace root\default -computername $computer -Credential $Credentials
			}
			else
			{
				$regProv = Get-Wmiobject -list "StdRegProv" -namespace root\default -computername $computer
			}
			
			[System.Collections.ArrayList]$VersionList = New-Object -TypeName System.Collections.ArrayList
			[System.Collections.ArrayList]$PathList = New-Object -TypeName System.Collections.ArrayList
			[System.Collections.ArrayList]$PackageList = New-Object -TypeName System.Collections.ArrayList
			[System.Collections.ArrayList]$ClickToRunPathList = New-Object -TypeName System.Collections.ArrayList
			[System.Collections.ArrayList]$ConfigItemList = New-Object -TypeName System.Collections.ArrayList
			$ClickToRunList = new-object PSObject[] 0;
			
			foreach ($regKey in $officeKeys)
			{
				$officeVersion = $regProv.EnumKey($HKLM, $regKey)
				foreach ($key in $officeVersion.sNames)
				{
					if ($key -match "\d{2}\.\d")
					{
						if (!$VersionList.Contains($key))
						{
							$AddItem = $VersionList.Add($key)
						}
						
						$path = join-path $regKey $key
						
						$configPath = join-path $path "Common\Config"
						$configItems = $regProv.EnumKey($HKLM, $configPath)
						if ($configItems)
						{
							foreach ($configId in $configItems.sNames)
							{
								if ($configId)
								{
									$Add = $ConfigItemList.Add($configId.ToUpper())
								}
							}
						}
						
						$cltr = New-Object -TypeName PSObject
						$cltr | Add-Member -MemberType NoteProperty -Name InstallPath -Value ""
						$cltr | Add-Member -MemberType NoteProperty -Name UpdatesEnabled -Value $false
						$cltr | Add-Member -MemberType NoteProperty -Name UpdateUrl -Value ""
						$cltr | Add-Member -MemberType NoteProperty -Name StreamingFinished -Value $false
						$cltr | Add-Member -MemberType NoteProperty -Name Platform -Value ""
						$cltr | Add-Member -MemberType NoteProperty -Name ClientCulture -Value ""
						
						$packagePath = join-path $path "Common\InstalledPackages"
						$clickToRunPath = join-path $path "ClickToRun\Configuration"
						$virtualInstallPath = $regProv.GetStringValue($HKLM, $clickToRunPath, "InstallationPath").sValue
						
						[string]$officeLangResourcePath = join-path  $path "Common\LanguageResources"
						$mainLangId = $regProv.GetDWORDValue($HKLM, $officeLangResourcePath, "SKULanguage").uValue
						if ($mainLangId)
						{
							$mainlangCulture = [globalization.cultureinfo]::GetCultures("allCultures") | where { $_.LCID -eq $mainLangId }
							if ($mainlangCulture)
							{
								$cltr.ClientCulture = $mainlangCulture.Name
							}
						}
						
						[string]$officeLangPath = join-path  $path "Common\LanguageResources\InstalledUIs"
						$langValues = $regProv.EnumValues($HKLM, $officeLangPath);
						if ($langValues)
						{
							foreach ($langValue in $langValues)
							{
								$langCulture = [globalization.cultureinfo]::GetCultures("allCultures") | where { $_.LCID -eq $langValue }
							}
						}
						
						if ($virtualInstallPath)
						{
							
						}
						else
						{
							$clickToRunPath = join-path $regKey "ClickToRun\Configuration"
							$virtualInstallPath = $regProv.GetStringValue($HKLM, $clickToRunPath, "InstallationPath").sValue
						}
						
						if ($virtualInstallPath)
						{
							if (!$ClickToRunPathList.Contains($virtualInstallPath.ToUpper()))
							{
								$AddItem = $ClickToRunPathList.Add($virtualInstallPath.ToUpper())
							}
							
							$cltr.InstallPath = $virtualInstallPath
							$cltr.StreamingFinished = $regProv.GetStringValue($HKLM, $clickToRunPath, "StreamingFinished").sValue
							$cltr.UpdatesEnabled = $regProv.GetStringValue($HKLM, $clickToRunPath, "UpdatesEnabled").sValue
							$cltr.UpdateUrl = $regProv.GetStringValue($HKLM, $clickToRunPath, "UpdateUrl").sValue
							$cltr.Platform = $regProv.GetStringValue($HKLM, $clickToRunPath, "Platform").sValue
							$cltr.ClientCulture = $regProv.GetStringValue($HKLM, $clickToRunPath, "ClientCulture").sValue
							$ClickToRunList += $cltr
						}
						
						$packageItems = $regProv.EnumKey($HKLM, $packagePath)
						$officeItems = $regProv.EnumKey($HKLM, $path)
						
						foreach ($itemKey in $officeItems.sNames)
						{
							$itemPath = join-path $path $itemKey
							$installRootPath = join-path $itemPath "InstallRoot"
							
							$filePath = $regProv.GetStringValue($HKLM, $installRootPath, "Path").sValue
							if (!$PathList.Contains($filePath))
							{
								$AddItem = $PathList.Add($filePath)
							}
						}
						
						foreach ($packageGuid in $packageItems.sNames)
						{
							$packageItemPath = join-path $packagePath $packageGuid
							$packageName = $regProv.GetStringValue($HKLM, $packageItemPath, "").sValue
							
							if (!$PackageList.Contains($packageName))
							{
								if ($packageName)
								{
									$AddItem = $PackageList.Add($packageName.Replace(' ', '').ToLower())
								}
							}
						}
						
					}
				}
			}
			
			
			
			foreach ($regKey in $installKeys)
			{
				$keyList = new-object System.Collections.ArrayList
				$keys = $regProv.EnumKey($HKLM, $regKey)
				
				foreach ($key in $keys.sNames)
				{
					$path = join-path $regKey $key
					$installPath = $regProv.GetStringValue($HKLM, $path, "InstallLocation").sValue
					if (!($installPath)) { continue }
					if ($installPath.Length -eq 0) { continue }
					
					$buildType = "64-Bit"
					if ($osArchitecture -eq "32-bit")
					{
						$buildType = "32-Bit"
					}
					
					if ($regKey.ToUpper().Contains("Wow6432Node".ToUpper()))
					{
						$buildType = "32-Bit"
					}
					
					if ($key -match "{.{8}-.{4}-.{4}-1000-0000000FF1CE}")
					{
						$buildType = "64-Bit"
					}
					
					if ($key -match "{.{8}-.{4}-.{4}-0000-0000000FF1CE}")
					{
						$buildType = "32-Bit"
					}
					
					if ($modifyPath)
					{
						if ($modifyPath.ToLower().Contains("platform=x86"))
						{
							$buildType = "32-Bit"
						}
						
						if ($modifyPath.ToLower().Contains("platform=x64"))
						{
							$buildType = "64-Bit"
						}
					}
                    
                    $primaryOfficeProduct = $false
                    $officeProduct = $false
                    foreach ($officeInstallPath in $PathList)
                    {
                        if ($officeInstallPath)
                        {
                            try
                            {
                                $installReg = "^" + $installPath.Replace('\', '\\')
                                $installReg = $installReg.Replace('(', '\(')
                                $installReg = $installReg.Replace(')', '\)')
                                if ($officeInstallPath -match $installReg) { $officeProduct = $true }
                            }
                            catch { }
                        }
                    }
                    
                    if (!$officeProduct) { continue };
					
					$name = $regProv.GetStringValue($HKLM, $path, "DisplayName").sValue
					
					if ($ConfigItemList.Contains($key.ToUpper()) -and $name.ToUpper().Contains("MICROSOFT OFFICE") -and $name.ToUpper() -notlike "*MUI*" -and $name.ToUpper() -notlike "*VISIO*" -and $name.ToUpper() -notlike "*PROJECT*")
					{
						$primaryOfficeProduct = $true
					}
					
					$clickToRunComponent = $regProv.GetDWORDValue($HKLM, $path, "ClickToRunComponent").uValue
					$uninstallString = $regProv.GetStringValue($HKLM, $path, "UninstallString").sValue
					if (!($clickToRunComponent))
					{
						if ($uninstallString)
						{
							if ($uninstallString.Contains("OfficeClickToRun"))
							{
								$clickToRunComponent = $true
							}
						}
					}
					
					$modifyPath = $regProv.GetStringValue($HKLM, $path, "ModifyPath").sValue
					$version = $regProv.GetStringValue($HKLM, $path, "DisplayVersion").sValue
					
					$cltrUpdatedEnabled = $NULL
					$cltrUpdateUrl = $NULL
					$clientCulture = $NULL;
					
					[string]$clickToRun = $false
					
					if ($clickToRunComponent)
					{
						$clickToRun = $true
						if ($name.ToUpper().Contains("MICROSOFT OFFICE"))
						{
							$primaryOfficeProduct = $true
						}
						
						foreach ($cltr in $ClickToRunList)
						{
							if ($cltr.InstallPath)
							{
								if ($cltr.InstallPath.ToUpper() -eq $installPath.ToUpper())
								{
									$cltrUpdatedEnabled = $cltr.UpdatesEnabled
									$cltrUpdateUrl = $cltr.UpdateUrl
									if ($cltr.Platform -eq 'x64')
									{
										$buildType = "64-Bit"
									}
									if ($cltr.Platform -eq 'x86')
									{
										$buildType = "32-Bit"
									}
									$clientCulture = $cltr.ClientCulture
								}
							}
						}
					}
					
					if (!$primaryOfficeProduct)
					{
						if (!$ShowAllInstalledProducts)
						{
							continue
						}
					}
					
					$object = New-Object PSObject -Property @{
						DisplayName   = $name; Version = $version; InstallPath = $installPath; ClickToRun = $clickToRun;
						Bitness	      = $buildType; ComputerName = $computer; ClickToRunUpdatesEnabled = $cltrUpdatedEnabled; ClickToRunUpdateUrl = $cltrUpdateUrl;
						ClientCulture = $clientCulture
					}
					$object | Add-Member MemberSet PSStandardMembers $PSStandardMembers
					$results += $object
					
				}
			}
			
		}
		
		$results = Get-Unique -InputObject $results
		
		return $results;
	}
	
}

#endregion

#region Function MinimizeAllWindows


#endregion

#region UNINSTALL - FUNCTION TO ASSIST FINDING UNINSTALL STRINGS, MSI CODES, DISPLAY NAMES OF INSTALLED APPLICATIONS
## function to assist finding uninstall strings, msi codes, display names of installed applications
# paste into powershell window (or save in (powershell profile)[http://www.howtogeek.com/50236/customizing-your-powershell-profile/]
# usage once loaded: 'Get-Uninstaller chrome'
function Get-Uninstaller
{
    [CmdletBinding()]
    param (
        [Parameter(Mandatory = $true)]
        [ValidateNotNullOrEmpty()]
        [string]$Name
    )
    
    $local_key = 'HKCU:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*'
    $machine_key32 = 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\*'
    $machine_key64 = 'HKLM:\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*'
    
    $keys = @($local_key, $machine_key32, $machine_key64)
    
    Get-ItemProperty -Path $keys -ErrorAction 'SilentlyContinue' | ?{ ($_.DisplayName -like "*$Name*") -or ($_.PsChildName -like "*$Name*") } | Select-Object PsPath, DisplayVersion, DisplayName, UninstallString, InstallSource, QuietUninstallString
}
## end of function
#endregion

#region FORCE INSTALL/UNINSTALL OF AVAILABLE SOFTWARE IN SOFTWARE CENTER THROUGH CIM/WMI ON A REMOTE CLIENT
##Powershell Function to Install and uninstall applications in software center on remote computer with the help of CIM. 
#Great way to save time and not needing to actually go in to software center on a remote machine when you want to perform a Install or uninstall of a specific appl
Function Trigger-AppInstallation
{
    
    Param
    (
        #[String][Parameter(Mandatory=$False, Position=1)] $Computername,
        [String][Parameter(Mandatory = $True, Position = 2)]
        $AppName,
        [ValidateSet("Install", "Uninstall")]
        [String][Parameter(Mandatory = $True, Position = 3)]
        $Method
    )
    
    Begin
    {
        
        $Application = (Get-CimInstance -ClassName CCM_Application -Namespace "root\ccm\clientSDK" | Where-Object { $_.Name -eq $AppName })
        
        $Args = @{
            EnforcePreference = [UINT32] 0
            Id                = "$($Application.id)"
            IsMachineTarget   = $Application.IsMachineTarget
            IsRebootIfNeeded  = $False
            Priority          = 'High'
            Revision          = "$($Application.Revision)"
        }
        
    }
    
    Process
    {
        
        Invoke-CimMethod -Namespace "root\ccm\clientSDK" -ClassName CCM_Application -MethodName $Method -Arguments $Args
        
    }
    
    End { }
    
}
## end of function
#endregion

#region ADD TO THE WINDOWS PATH ENVIRONMENT VARIABLE
## To use this to add to the PATH in your Deploy-Application.ps1 add $addPath = 'C:\TopSecret\Bin'
## To use this to remove a PATH in your Deploy-Application.ps1 add $removePath = 'C:\TopSecret\Bin'

Function Set-PathVariable
{
    param (
        [string]$AddPath,
        [string]$RemovePath
    )
    $regexPaths = @()
    if ($PSBoundParameters.Keys -contains 'AddPath')
    {
        $regexPaths += [regex]::Escape($AddPath)
    }
    
    if ($PSBoundParameters.Keys -contains 'RemovePath')
    {
        $regexPaths += [regex]::Escape($RemovePath)
    }
    
    $arrPath = $env:Path -split ';'
    foreach ($path in $regexPaths)
    {
        $arrPath = $arrPath | Where-Object { $_ -notMatch "^$path\\?" }
    }
    $env:Path = ($arrPath + $addPath) -join ';'
}
## end of function
#endregion

#region FIND AND REPLACE TEXT IN A FILE
function Find-InTextFile
{
    <#
    .SYNOPSIS
        Performs a find (or replace) on a string in a text file or files.
    .EXAMPLE
        PS> Find-InTextFile -FilePath 'C:\MyFile.txt' -Find 'water' -Replace 'wine'
    
        Replaces all instances of the string 'water' into the string 'wine' in
        'C:\MyFile.txt'.
    .EXAMPLE
        PS> Find-InTextFile -FilePath 'C:\MyFile.txt' -Find 'water'
    
        Finds all instances of the string 'water' in the file 'C:\MyFile.txt'.
    .PARAMETER FilePath
        The file path of the text file you'd like to perform a find/replace on.
    .PARAMETER Find
        The string you'd like to replace.
    .PARAMETER Replace
        The string you'd like to replace your 'Find' string with.
    .PARAMETER NewFilePath
        If a new file with the replaced the string needs to be created instead of replacing
        the contents of the existing file use this param to create a new file.
    .PARAMETER Force
        If the NewFilePath param is used using this param will overwrite any file that
        exists in NewFilePath.
    #>
    [CmdletBinding(DefaultParameterSetName = 'NewFile')]
    [OutputType()]
    param (
        [Parameter(Mandatory = $true)]
        [ValidateScript({ Test-Path -Path $_ -PathType 'Leaf' })]
        [string[]]$FilePath,
        [Parameter(Mandatory = $true)]
        [string]$Find,
        [Parameter()]
        [string]$Replace,
        [Parameter(ParameterSetName = 'NewFile')]
        [ValidateScript({ Test-Path -Path ($_ | Split-Path -Parent) -PathType 'Container' })]
        [string]$NewFilePath,
        [Parameter(ParameterSetName = 'NewFile')]
        [switch]$Force
    )
    begin
    {
        $Find = [regex]::Escape($Find)
    }
    process
    {
        try
        {
            foreach ($File in $FilePath)
            {
                if ($Replace)
                {
                    if ($NewFilePath)
                    {
                        if ((Test-Path -Path $NewFilePath -PathType 'Leaf') -and $Force.IsPresent)
                        {
                            Remove-Item -Path $NewFilePath -Force
                            (Get-Content $File) -replace $Find, $Replace | Add-Content -Path $NewFilePath -Force
                        }
                        elseif ((Test-Path -Path $NewFilePath -PathType 'Leaf') -and !$Force.IsPresent)
                        {
                            Write-Warning "The file at '$NewFilePath' already exists and the -Force param was not used"
                        }
                        else
                        {
                            (Get-Content $File) -replace $Find, $Replace | Add-Content -Path $NewFilePath -Force
                        }
                    }
                    else
                    {
                        (Get-Content $File) -replace $Find, $Replace | Add-Content -Path "$File.tmp" -Force
                        Remove-Item -Path $File
                        Move-Item -Path "$File.tmp" -Destination $File
                    }
                }
                else
                {
                    Select-String -Path $File -Pattern $Find
                }
            }
        }
        catch
        {
            Write-Error $_.Exception.Message
        }
    }
}
## end of function
#endregion

#region Removes files or directories reliably and synchronously
function Remove-FileSystemItem
{
  <#
  .SYNOPSIS
    Removes files or directories reliably and synchronously.

  .DESCRIPTION
    Removes files and directories, ensuring reliable and synchronous
    behavior across all supported platforms.

    The syntax is a subset of what Remove-Item supports; notably,
    -Include / -Exclude and -Force are NOT supported; -Force is implied.

    As with Remove-Item, passing -Recurse is required to avoid a prompt when 
    deleting a non-empty directory.

    IMPORTANT:
      * On Unix platforms, this function is merely a wrapper for Remove-Item, 
        where the latter works reliably and synchronously, but on Windows a 
        custom implementation must be used to ensure reliable and synchronous 
        behavior. See https://github.com/PowerShell/PowerShell/issues/8211

    * On Windows:
      * The *parent directory* of a directory being removed must be 
        *writable* for the synchronous custom implementation to work.
      * The custom implementation is also applied when deleting 
         directories on *network drives*.

    * If an indefinitely *locked* file or directory is encountered, removal is aborted.
      By contrast, files opened with FILE_SHARE_DELETE / 
      [System.IO.FileShare]::Delete on Windows do NOT prevent removal, 
      though they do live on under a temporary name in the parent directory 
      until the last handle to them is closed.

    * Hidden files and files with the read-only attribute:
      * These are *quietly removed*; in other words: this function invariably
        behaves like `Remove-Item -Force`.
      * Note, however, that in order to target hidden files / directories
        as *input*, you must specify them as a *literal* path, because they
        won't be found via a wildcard expression.

    * The reliable custom implementation on Windows comes at the cost of
      decreased performance.

  .EXAMPLE
    Remove-FileSystemItem C:\tmp -Recurse

    Synchronously removes directory C:\tmp and all its content.
  #>
    [CmdletBinding(SupportsShouldProcess, ConfirmImpact = 'Medium', DefaultParameterSetName = 'Path', PositionalBinding = $false)]
    param (
        [Parameter(ParameterSetName = 'Path', Mandatory, Position = 0, ValueFromPipeline, ValueFromPipelineByPropertyName)]
        [string[]]$Path
         ,
        [Parameter(ParameterSetName = 'Literalpath', ValueFromPipelineByPropertyName)]
        [Alias('PSPath')]
        [string[]]$LiteralPath
         ,
        [switch]$Recurse
    )
    begin
    {
        # !! Workaround for https://github.com/PowerShell/PowerShell/issues/1759
        if ($ErrorActionPreference -eq [System.Management.Automation.ActionPreference]::Ignore) { $ErrorActionPreference = 'Ignore' }
        $targetPath = ''
        $yesToAll = $noToAll = $false
        function trimTrailingPathSep([string]$itemPath)
        {
            if ($itemPath[-1] -in '\', '/')
            {
                # Trim the trailing separator, unless the path is a root path such as '/' or 'c:\'
                if ($itemPath.Length -gt 1 -and $itemPath -notmatch '^[^:\\/]+:.$')
                {
                    $itemPath = $itemPath.Substring(0, $itemPath.Length - 1)
                }
            }
            $itemPath
        }
        function getTempPathOnSameVolume([string]$itemPath, [string]$tempDir)
        {
            if (-not $tempDir) { $tempDir = [IO.Path]::GetDirectoryName($itemPath) }
            [IO.Path]::Combine($tempDir, [IO.Path]::GetRandomFileName())
        }
        function syncRemoveFile([string]$filePath, [string]$tempDir)
        {
            # Clear the ReadOnly attribute, if present.
            if (($attribs = [IO.File]::GetAttributes($filePath)) -band [System.IO.FileAttributes]::ReadOnly)
            {
                [IO.File]::SetAttributes($filePath, $attribs -band -bnot [System.IO.FileAttributes]::ReadOnly)
            }
            $tempPath = getTempPathOnSameVolume $filePath $tempDir
            [IO.File]::Move($filePath, $tempPath)
            [IO.File]::Delete($tempPath)
        }
        function syncRemoveDir([string]$dirPath, [switch]$recursing)
        {
            if (-not $recursing) { $dirPathParent = [IO.Path]::GetDirectoryName($dirPath) }
            # Clear the ReadOnly attribute, if present.
            # Note: [IO.File]::*Attributes() is also used for *directories*; [IO.Directory] doesn't have attribute-related methods.
            if (($attribs = [IO.File]::GetAttributes($dirPath)) -band [System.IO.FileAttributes]::ReadOnly)
            {
                [IO.File]::SetAttributes($dirPath, $attribs -band -bnot [System.IO.FileAttributes]::ReadOnly)
            }
            # Remove all children synchronously.
            $isFirstChild = $true
            foreach ($item in [IO.directory]::EnumerateFileSystemEntries($dirPath))
            {
                if (-not $recursing -and -not $Recurse -and $isFirstChild)
                {
                    # If -Recurse wasn't specified, prompt for nonempty dirs.
                    $isFirstChild = $false
                    # Note: If -Confirm was also passed, this prompt is displayed *in addition*, after the standard $PSCmdlet.ShouldProcess() prompt.
                    #       While Remove-Item also prompts twice in this scenario, it shows the has-children prompt *first*.
                    if (-not $PSCmdlet.ShouldContinue("The item at '$dirPath' has children and the -Recurse switch was not specified. If you continue, all children will be removed with the item. Are you sure you want to continue?", 'Confirm', ([ref]$yesToAll), ([ref]$noToAll))) { return }
                }
                $itemPath = [IO.Path]::Combine($dirPath, $item)
                ([ref]$targetPath).Value = $itemPath
                if ([IO.Directory]::Exists($itemPath))
                {
                    syncremoveDir $itemPath -recursing
                }
                else
                {
                    syncremoveFile $itemPath $dirPathParent
                }
            }
            # Finally, remove the directory itself synchronously.
            ([ref]$targetPath).Value = $dirPath
            $tempPath = getTempPathOnSameVolume $dirPath $dirPathParent
            [IO.Directory]::Move($dirPath, $tempPath)
            [IO.Directory]::Delete($tempPath)
        }
    }
    
    process
    {
        $isLiteral = $PSCmdlet.ParameterSetName -eq 'LiteralPath'
        if ($env:OS -ne 'Windows_NT')
        {
            # Unix: simply pass through to Remove-Item, which on Unix works reliably and synchronously
            Remove-Item @PSBoundParameters
        }
        else
        {
            # Windows: use synchronous custom implementation
            foreach ($rawPath in ($Path, $LiteralPath)[$isLiteral])
            {
                # Resolve the paths to full, filesystem-native paths.
                try
                {
                    # !! Convert-Path does find hidden items via *literal* paths, but not via *wildcards* - and it has no -Force switch (yet)
                    # !! See https://github.com/PowerShell/PowerShell/issues/6501
                    $resolvedPaths = if ($isLiteral) { Convert-Path -ErrorAction Stop -LiteralPath $rawPath }
                    else { Convert-Path -ErrorAction Stop -path $rawPath }
                }
                catch
                {
                    Write-Error $_ # relay error, but in the name of this function
                    continue
                }
                try
                {
                    $isDir = $false
                    foreach ($resolvedPath in $resolvedPaths)
                    {
                        # -WhatIf and -Confirm support.
                        if (-not $PSCmdlet.ShouldProcess($resolvedPath)) { continue }
                        if ($isDir = [IO.Directory]::Exists($resolvedPath))
                        {
                            # dir.
                            # !! A trailing '\' or '/' causes directory removal to fail ("in use"), so we trim it first.
                            syncRemoveDir (trimTrailingPathSep $resolvedPath)
                        }
                        elseif ([IO.File]::Exists($resolvedPath))
                        {
                            # file
                            syncRemoveFile $resolvedPath
                        }
                        else
                        {
                            Throw "Not a file-system path or no longer extant: $resolvedPath"
                        }
                    }
                }
                catch
                {
                    if ($isDir)
                    {
                        $exc = $_.Exception
                        if ($exc.InnerException) { $exc = $exc.InnerException }
                        if ($targetPath -eq $resolvedPath)
                        {
                            Write-Error "Removal of directory '$resolvedPath' failed: $exc"
                        }
                        else
                        {
                            Write-Error "Removal of directory '$resolvedPath' failed, because its content could not be (fully) removed: $targetPath`: $exc"
                        }
                    }
                    else
                    {
                        Write-Error $_ # relay error, but in the name of this function
                    }
                    continue
                }
            }
        }
    }
}
## end of function
#endregion

##*===============================================
##* END FUNCTION LISTINGS
##*===============================================

##*===============================================
##* SCRIPT BODY
##*===============================================

If ($scriptParentPath) {
	Write-Log -Message "Script [$($MyInvocation.MyCommand.Definition)] dot-source invoked by [$(((Get-Variable -Name MyInvocation).Value).ScriptName)]" -Source $appDeployToolkitExtName
} Else {
	Write-Log -Message "Script [$($MyInvocation.MyCommand.Definition)] invoked directly" -Source $appDeployToolkitExtName
}

##*===============================================
##* END SCRIPT BODY
##*===============================================
